# Notum AHI — Usage Guide

Copyright © 2026 Notum Robotics. All rights reserved.
Licensed under the MIT License. See [LICENSE](#license) below.

---

## Table of Contents

1. [Getting Started](#getting-started)
2. [File Structure](#file-structure)
3. [HTML Boilerplate](#html-boilerplate)
4. [Components](#components)
   - [Buttons](#buttons)
   - [Panels & Corner Anchors](#panels--corner-anchors)
   - [Toggle Groups](#toggle-groups)
   - [Segmented Sliders](#segmented-sliders)
   - [Vertical Segmented Sliders](#vertical-segmented-sliders)
   - [Segmented Bars](#segmented-bars)
   - [Vertical Segmented Bars](#vertical-segmented-bars)
   - [Numeric Steppers](#numeric-steppers)
   - [Active State Cards](#active-state-cards)
   - [Icon Toggle Buttons](#icon-toggle-buttons)
   - [Status Readouts](#status-readouts)
   - [Sub-menu Items](#sub-menu-items)
   - [Dialogs](#dialogs)
5. [Auto-Fit Grid](#auto-fit-grid)
6. [Card Size Variants](#card-size-variants)
7. [Component Properties (nComp)](#component-properties-ncomp)
   - [Progress Bar](#progress-bar)
   - [Status Pip](#status-pip)
   - [Active / Inactive State](#active--inactive-state)
8. [Dynamic Layout (nDynamic)](#dynamic-layout-ndynamic)
   - [Quick Start](#quick-start)
   - [Control Schema](#control-schema)
   - [Configuration](#configuration)
   - [API](#api)
   - [How It Works](#how-it-works)
8. [Interactive Layout (nInteractive)](#interactive-layout-ninteractive)
   - [Quick Start (Interactive)](#quick-start-interactive)
   - [Interaction Model](#interaction-model)
   - [Context Menu](#context-menu)
   - [Drag & Drop](#drag--drop)
   - [Lock System](#lock-system)
   - [Mute System](#mute-system)
   - [Collision Detection](#collision-detection)
   - [CSS Classes Reference](#css-classes-reference)
   - [Configuration (Interactive)](#configuration-interactive)
   - [API (Interactive)](#api-interactive)
9. [Procedural Audio (nbeep)](#procedural-audio-nbeep)
   - [Basic Usage](#basic-usage)
   - [Soundscapes](#soundscapes)
   - [Configuration (Audio)](#configuration-audio)
   - [Automatic Audio Integration](#automatic-audio-integration)
   - [Manual Audio Triggers](#manual-audio-triggers)
10. [In-Grid Notifications (nNotify)](#in-grid-notifications-nnotify)
11. [Interaction Feedback](#interaction-feedback)
    - [Flash Feedback (flashOutline)](#flash-feedback-flashoutline)
    - [Extended Lockout](#extended-lockout)
12. [Icons](#icons)
13. [Color Reference](#color-reference)
14. [License](#license)

---

## Getting Started

Include the three CSS files and two JS files in your HTML. All assets are self-hosted — no CDNs required.

```html
<!-- CSS -->
<link rel="stylesheet" href="css/phosphor.css">
<link rel="stylesheet" href="css/style.css">
<link rel="stylesheet" href="css/notum.css">

<!-- JS (load at end of body) -->
<script src="js/nbeep.js"></script>
<script src="js/notum.js"></script>
```

**Load order matters:** `nbeep.js` must load before `notum.js` because the demo script checks for the `nbeep` global.

For the dynamic grid engine and interactive editor, add:

```html
<script src="js/nDynamic.js"></script>
<script src="js/nInteractive.js"></script>  <!-- optional: only if you need edit mode -->
```

`nDynamic.js` must load before `nInteractive.js`.

---

## File Structure

```
notum_design_system/
├── index.html            # Homepage (AHI dashboard + scenarios)
├── components.html       # Component showcase page
├── dynamic.html          # Dynamic grid layout demo
├── DESIGN_GUDELINES.md   # Full design specification
├── USAGE.md              # This file
├── README.md             # Project overview
├── LICENSE               # MIT License
├── css/
│   ├── phosphor.css      # Phosphor icon font stylesheet
│   ├── style.css         # Base / reset styles + card size variants
│   ├── notum.css         # All Notum component styles
│   ├── subpage.css       # Shared sub-page viewport layout
│   └── demo.css          # Styles specific to component showcase page
├── fonts/
│   ├── rajdhani-400.woff2
│   ├── rajdhani-500.woff2
│   ├── rajdhani-600.woff2
│   ├── rajdhani-700.woff2
│   ├── ibm-plex-mono-400.woff2
│   ├── ibm-plex-mono-500.woff2
│   └── phosphor-regular.woff2
└── js/
    ├── nUtils.js         # Shared utilities (escHtml, flashOutline, etc.)
    ├── nComp.js          # Component properties (progress, status, active)
    ├── nCatalog.js       # Shared control catalog (GRID_CATALOG)
    ├── nbeep.js          # Procedural audio engine (standalone)
    ├── nDynamic.js       # Dynamic viewport-filling grid engine
    ├── nInteractive.js   # Interactive layout editor (wraps nDynamic)
    ├── nNotify.js        # In-grid notification system
    ├── notum.js          # Component page behaviors & wiring
    └── ahi/
        ├── ahi.js            # Core AHI API (tool calls, rendering)
        ├── ahi-protocol.js   # JSON-RPC 2.0 message handler
        ├── ahi-transport-ws.js # WebSocket + PostMessage transport adapters
        ├── ahi-flow.js       # Multi-step wizard/flow engine
        └── ahi-schema.json   # JSON Schema for AHI control types
```

---

## HTML Boilerplate

Minimal page setup — see `components.html` or `dynamic.html` for full working examples.

The dialog overlay element (`dialog-overlay` containing `dialog-box`) must be present in the DOM for dialogs to work.

---

## Components

### Buttons

Four style variants. Always add `data-flash` for press feedback.

- `action-btn` — Default style
- `action-btn primary` — Primary action
- `action-btn warning` — Warning action
- `action-btn danger` — Destructive action

Add an icon with: `<i class="ph ph-power"></i>`

**With audio:** Add `data-nbeep="your_identifier"` to produce a specific sound.

**With dialog:** Add a `data-dialog` attribute containing a JSON dialog definition object. Clicking the button automatically opens the dialog via `showDialog()`.

---

### Panels & Corner Anchors

Every panel must include all four corner anchor glyphs (`corner-tl`, `corner-tr`, `corner-bl`, `corner-br`) using the Unicode bracket characters.

---

### Toggle Groups

Mutually exclusive option selector inside a `toggle-group` container. One `tg-option` must have the `active` class. Reading the value: query `.tg-option.active` or listen for clicks on `opt.dataset.val`.

---

### Segmented Sliders

Interactive slider built from discrete segments. Requires `data-value`, `data-max`, and `data-color` attributes on a `seg-slider` element.

Color options: `accent` (cyan), `amber`, `danger` (red).

Reading value changes: Use a `MutationObserver` on the `data-value` attribute.

**Pitch-varying audio:** Sliders produce pitch-scaled `nbeep('adjust')` sounds — lower pitch at the left, higher at the right. The pitch multiplier is `2^((pct - 0.5) * 2)`.

---

### Segmented Bars

Static (non-interactive) display version of the slider with built-in animations:

- **Partially filled bars:** A traveling dim highlight (`brightness(0.75)`) sweeps across filled segments every 120ms. The sweep direction reflects the last value change — left-to-right (or bottom-to-top for vertical) when increasing, right-to-left (top-to-bottom) when decreasing.
- **Full bars** (value >= max): A double-blink pattern (50ms on/off × 2 flashes, 350ms pause, loop).
- **Value transitions:** When bar values change via `nDynamic.updateSegBar()`, animated transitions provide feedback. On decrease, removed segments fade out (opacity → 0, 150ms CSS transition). On increase, new segments flash briefly brighter (`brightness(1.4)`, 120ms). Travel/blink animation restarts after either transition.

---

### Vertical Segmented Sliders

Interactive vertical slider variant. Same attributes as horizontal sliders but displayed as a bottom-to-top column.

```html
<div class="panel v-gauge-panel" style="min-width:70px; height:200px">
    <span class="corner corner-tl">⌜</span>
    <span class="corner corner-tr">⌝</span>
    <span class="corner corner-bl">⌞</span>
    <span class="corner corner-br">⌟</span>
    <div class="panel-label">VOLUME</div>
    <div class="seg-slider-v" id="vslider-volume"
         data-value="6" data-max="10" data-color="accent"
         style="flex:1; width:28px"></div>
    <span class="slider-readout">[ 60% ]</span>
</div>
```

**Key differences from horizontal:**
- Uses `seg-slider-v` class instead of `seg-slider`
- Rendered with CSS `column-reverse` — visual bottom = index 0 (low value), top = max (high value)
- Drag interaction maps Y position: clicking near the top → high value, near the bottom → low value
- Same `data-value`, `data-max`, `data-color` attributes
- Same pitch-scaling audio behavior on drag
- Recommended container: 28px width, flexible height

---

### Vertical Segmented Bars

Read-only vertical bar variant. Same animations as horizontal bars but displayed bottom-to-top.

```html
<div class="panel v-gauge-panel" style="min-width:70px; height:200px">
    <span class="corner corner-tl">⌜</span>
    <span class="corner corner-tr">⌝</span>
    <span class="corner corner-bl">⌞</span>
    <span class="corner corner-br">⌟</span>
    <div class="panel-label">SIGNAL</div>
    <div class="seg-bar-v" id="vbar-signal"
         data-value="14" data-max="20" data-color="accent"
         style="flex:1; width:14px"></div>
    <span class="slider-readout">[ 70% ]</span>
</div>
```

**Key differences from horizontal:**
- Uses `seg-bar-v` class instead of `seg-bar`
- Rendered with CSS `column-reverse`
- Same `data-value`, `data-max`, `data-color` attributes
- Same travel/blink animations
- Recommended container: 14px width, flexible height

---

### Numeric Steppers

Increment/decrement control with `stepper-btn` elements carrying `data-dir="-1"` or `data-dir="+1"`. Value is clamped to a minimum of 0 (no upper bound).

---

### Active State Cards

Toggle-able state cards with icon, name, and status indicator. Use class `is-on` or `is-off`. The `data-on` / `data-off` attributes define the label text for each state.

---

### Status Readouts

Key-value data display using `status-row` with `status-label` and `status-val`. Color classes for values: `accent` (cyan), `amber`, `danger`.

---

### Icon Toggle Buttons

Compact icon-only toggles that flip between on/off states on click. Use inside panels or inline layouts.

```html
<div class="icon-btn is-on icon-toggle" data-flash><i class="ph ph-power"></i></div>
<div class="icon-btn is-off icon-toggle" data-flash><i class="ph ph-bluetooth"></i></div>
```

**Required classes:** `icon-btn`, `icon-toggle`, and initial state: `is-on` or `is-off`.

**Behavior:**
- Click toggles between `is-on` and `is-off` CSS classes
- `data-flash` triggers the standard strobe feedback on click
- The `is-on` state shows the icon at full opacity; `is-off` dims it
- `notum.js` wires click handlers automatically when the page loads

---

### Sub-menu Items

List-style navigation items with icon, name, value, and chevron. Support color variants for status indication.

```html
<div class="submenu">
    <div class="submenu-item" data-flash>
        <i class="ph ph-wifi-high"></i>
        <div class="submenu-info">
            <span class="submenu-name">WI-FI</span>
            <span class="submenu-value">[ CONNECTED ]</span>
        </div>
        <span class="submenu-chevron">›</span>
    </div>
    <div class="submenu-item warning" data-flash>
        <i class="ph ph-cloud"></i>
        <div class="submenu-info">
            <span class="submenu-name">CLOUD SYNC</span>
            <span class="submenu-value amber">[ DEGRADED ]</span>
        </div>
        <span class="submenu-chevron">›</span>
    </div>
    <div class="submenu-item danger" data-flash>
        <i class="ph ph-plugs"></i>
        <div class="submenu-info">
            <span class="submenu-name">GATEWAY</span>
            <span class="submenu-value danger">[ OFFLINE ]</span>
        </div>
        <span class="submenu-chevron">›</span>
    </div>
</div>
```

**Structure:**
- `.submenu` — container
- `.submenu-item` — individual row (add `warning` or `danger` class for color variants)
- `.submenu-info` — wraps name and value
- `.submenu-name` — primary label
- `.submenu-value` — secondary status text (add `accent`, `amber`, or `danger` class for color)
- `.submenu-chevron` — right-aligned `›` glyph
- `data-flash` — enables click strobe feedback
- Audio: automatically triggers `nbeep('menu_' + name)` on click

---

### Dialogs

Dialogs are invoked programmatically via `showDialog(opts)` which returns a Promise.

**Dialog definition format:**
- `title` (string): Dialog title. Also used as base for the looping beep sound.
- `body` (string): Body text. Also passed to nbeep as secondary parameter.
- `buttons` (array): Array of `{ label, value, style? }` objects.
- `alarm` (boolean, optional): Whether to play the looping beep alarm. Default `true`. Pass `false` to suppress the alarm sound while keeping all other dialog behavior (overlay, corner strobe, etc.).

**Button styles:** omit `style` for default, or use `'primary'`, `'warning'`, or `'danger'`.

**Behavior:**
- Opens with a dithered overlay (instant, no animation).
- Dialog corner anchors strobe continuously (5 flashes at 50ms intervals, 200ms pause, loop).
- A looping `nbeep()` tone plays derived from the dialog's title + body text (unless `alarm: false`).
- Clicking a button fires its own `nbeep('dialog_btn_' + value)`, triggers flash feedback, then closes after 200ms.
- Clicking outside the dialog dismisses it and resolves the Promise with `null`.
- Call `closeDialog(value)` to close programmatically.

---

## Auto-Fit Grid

The framework includes a bin-packed auto-fit grid that renders a catalog of components into a responsive 2- or 4-column layout. The grid is populated programmatically from a `GRID_CATALOG` array in `notum.js`. Supported item types: `card`, `slider`, `toggle`, `button`, `stepper`, `bar`, `status`.

---

## Card Size Variants

Active state cards support nine `data-size` variants that scale from 1×1 up to 3×3. The same card markup is used for every size — CSS dynamically hides or rearranges elements to fit each variant.

| Size | Grid Span | Visible Elements | Use Case |
|---|---|---|---|
| `3x3` | 3 cols × 3 rows | Icon (64px), name, state text | Extra-large hero card |
| `3x2` | 3 cols × 2 rows | Icon (56px), name, state text | Wide landscape |
| `2x3` | 2 cols × 3 rows | Icon (56px), name, state (stacked vertical) | Tall portrait |
| `2x2` (default) | 2 cols × 2 rows | Icon, name, state text | Standard full card |
| `3x1` | 3 cols × 1 row | Icon, name only | Wide strip |
| `2x1` | 2 cols × 1 row | Icon, name only | Compact horizontal |
| `1x3` | 1 col × 3 rows | Icon, name, state (stacked vertical) | Tall pillar |
| `1x2` | 1 col × 2 rows | Icon, name (stacked vertical) | Compact vertical |
| `1x1` | 1 col × 1 row | Icon only | Minimal / toolbar |

Add a `data-size` attribute to any `.card` element. In the auto-grid catalog or nDynamic controls array, add a `size` property to card entries.

---

## Component Properties (nComp)

The `nComp` system provides a uniform API for attaching progress bars, status indicators, and active/inactive states to **any** DOM element. It is exposed globally as `window.nComp` by `notum.js`.

### Progress Bar

A 10-segment progress bar injected at the bottom of any element.

```js
// Partial progress (0–99): fills segments proportionally.
// A dim-highlight "travel" animation sweeps across filled segments every 120ms.
// Direction is automatic: increasing values sweep L→R, decreasing sweeps R→L.
nComp.progress(myElement, 50);   // 50% — 5 of 10 segments filled

// Full progress (100): all segments filled.
// Switches to a double-blink pattern (50ms on/off × 2 flashes, 350ms pause, loop).
nComp.progress(myElement, 100);

// Indeterminate (-1): no segments filled.
// A single bright segment scans left-to-right continuously (100ms per step).
nComp.progress(myElement, -1);

// Remove progress bar entirely
nComp.progress(myElement, null);
```

**Animation state machine:**

| Value | State | Animation | CSS attribute |
|---|---|---|---|
| `0`–`99` | Partial | Traveling dim-highlight across filled segs (120ms); direction follows last value change (L→R on increase, R→L on decrease) | `data-progress="<N>"` |
| `100` | Full | Double-blink on all segs (50ms flash × 2, 350ms pause) | `data-progress="100"` |
| `-1` | Indeterminate | Single bright segment scans in current direction (100ms); inherits direction from last determinate update | `data-progress="indeterminate"` |
| `null` | Removed | Bar element removed from DOM | attribute removed |

**CSS architecture:** The bar is a `.n-progress` div containing 10 `.n-seg` children. It is absolutely positioned at the bottom of the host element (which gets `position: relative` if static). Segments use the element's `data-color` for accent coloring.

### Status Pip

A 10×10px indicator dot positioned at the top-right corner of any element.

```js
nComp.status(myElement, 'ok');    // Green pip
nComp.status(myElement, 'warn');  // Amber pip
nComp.status(myElement, 'error'); // Red pip
nComp.status(myElement, 'busy');  // Cyan pip + CSS pulse animation
nComp.status(myElement, null);    // Remove pip
```

| Status | Color | Extra |
|---|---|---|
| `ok` | `#2ecc40` (green) | Static |
| `warn` | `var(--amber)` | Static |
| `error` | `var(--danger)` | Static |
| `busy` | `var(--accent)` (cyan) | Pulsing animation |

**CSS:** `.n-status-pip` span, absolutely positioned `top: 4px; right: 4px`.

### Active / Inactive State

Toggles a `data-active` attribute for visual state changes.

```js
nComp.active(myElement, true);   // Sets data-active="true"
nComp.active(myElement, false);  // Sets data-active="false"
nComp.active(myElement, null);   // Removes data-active attribute
```

**Combining properties:** Progress, status, and active can be used together on the same element:

```js
// Button with progress bar, status pip, and active state
nComp.progress(btn, 0);
nComp.status(btn, 'busy');
// ... during operation ...
nComp.progress(btn, 75);
// ... on completion ...
nComp.progress(btn, 100);
nComp.status(btn, 'ok');
// ... cleanup after delay ...
nComp.progress(btn, null);
nComp.status(btn, null);
```

---

## Dynamic Layout (nDynamic)

`nDynamic` is an intelligent viewport-filling grid layout engine. It arranges any number of controls into an optimal grid that fills the screen without scrolling. Controls are defined declaratively — the engine handles placement, sizing, and responsive adaptation.

**Demo pages:** `index.html` (AHI homepage, with nInteractive), `dynamic.html` (standalone)

### Quick Start

Include `nbeep.js` and `nDynamic.js`, then call:

```js
nDynamic.init('#my-grid', controlsArray, configObject);
```

### Control Schema

| Property | Type | Required | Description |
|---|---|---|---|
| `type` | string | Yes | `'card'`, `'slider'`, `'toggle'`, `'button'`, `'stepper'`, `'bar'`, `'status'` |
| `cols` | number | Yes | Column span (1–6) |
| `rows` | number | Yes | Row span (1–4) |
| `size` | string | | Size variant: `'3x3'`, `'3x2'`, `'2x3'`, `'2x2'`, `'3x1'`, `'2x1'`, `'1x3'`, `'1x2'`, `'1x1'` |
| `state` | string | | Card initial state: `'on'` or `'off'` |
| `icon` | string | | Phosphor icon class, e.g. `'ph-lock'` |
| `name` | string | | Card display name |
| `on`/`off` | string | | Card state labels (default: `'ON'`/`'OFF'`) |
| `label` | string | | Label for sliders, toggles, steppers, bars, buttons |
| `max` | number | | Max segments for sliders/bars (default: 10) |
| `value` | number | | Current value for sliders/bars/steppers (default: 0) |
| `color` | string | | `'accent'`, `'amber'`, or `'danger'` |
| `style` | string | | Button style: `''` (default), `'primary'`, `'warning'`, `'danger'` |
| `options` | array | | Toggle group options (array of strings) |
| `active` | number | | Toggle group active index (0-based) |
| `items` | array | | Status group rows: `[{ k, v, c }]` — k = label, v = value text, c = CSS color class |
| `dialog` | object | | Button dialog definition: `{ title, body, buttons: [{ label, value, style? }] }` |

### Configuration

| Config | Type | Default | Description |
|---|---|---|---|
| `cols` | number | auto | Force column count. Auto breakpoints: 2 (<600px), 4 (600–900px), 6 (>900px) — based on usable width after padding |
| `rowHeight` | number | auto | Row height in px. Auto: computed to fill viewport height, clamped to 40–120px |
| `gap` | number | `6` | Grid gap in px |
| `padding` | number | `16` | Container padding in px |
| `order` | number[] | — | Render order by control index. Controls not listed are appended in original order |
| `pinned` | object | — | Pin controls to `{ col, row }` grid positions (0-based). Pinned items use `grid-column: N / span C; grid-row: N / span R` for exact placement while preserving span |

### API

| Method | Description |
|---|---|
| `nDynamic.init(selectorOrElement, controlsArray, configObject)` | Initialize |
| `nDynamic.rebuild()` | Force rebuild (debounced via `requestAnimationFrame`) |
| `nDynamic.update(newControls, newConfig)` | Update controls and/or config, then rebuild (pass `null` to keep current) |
| `nDynamic.destroy()` | Tear down: remove grid, observers, listeners |
| `nDynamic.updateSegBar(barEl, newValue)` | Animate a `.seg-bar` element to a new value (fade-out on decrease, flash on increase) |
| `nDynamic.showDialog({ title, body, buttons })` | Open a dialog programmatically (returns a Promise) |
| `nDynamic.closeDialog(value)` | Close any open dialog |

### How It Works

1. Measures available viewport area (height from container top to viewport bottom).
2. Computes optimal column count (2/4/6 by width breakpoints) and row height to fill without scrolling. Row height is clamped to 40–120px. Available height floor is 200px.
3. Bin-packs controls using a row-occupancy bitmap for accurate dense-packing simulation. Pinned items are placed first at their specified positions; unpinned items fill remaining space using CSS Grid dense auto-flow.
4. Renders each control as a `.grid-cell` element with the correct `grid-column: span N` / `grid-row: span N`. Pinned items additionally get explicit start positions (`grid-column: N / span C`).
5. Wires all interactions (card toggle, slider drag, button flash + dialog, toggle select, stepper) automatically. Slider sounds use pitch-scaling. All sound calls check `isMuted()` — if the control's element has the `ni-muted` CSS class, sounds are suppressed.
6. Monitors container width and viewport height via `ResizeObserver` — rebuilds when dimensions change by >10px. A `window.resize` listener acts as fallback for height changes.

The engine is designed for zero-scroll dashboards: define your controls, call `init()`, and the layout adapts to any screen size.

---

## Interactive Layout (nInteractive)

`nInteractive` wraps `nDynamic` to add a hold-to-edit interaction layer with drag-and-drop reordering, context menus, and per-control lock / mute / resize options.

**Dependency:** `nDynamic.js` must be loaded before `nInteractive.js`.

### Quick Start (Interactive)

Include `nbeep.js`, `nDynamic.js`, and `nInteractive.js`, then call:

```js
nInteractive.init('#my-container', controlsArray, configObject);
```

See `index.html` (AHI homepage) for a complete working example with nInteractive enabled on a full control grid.

### Interaction Model

| Mode | Action | Effect |
|---|---|---|
| Normal | Tap / click | Control's native action (toggle, slider, button) |
| Normal | Hold 5 s | Enter edit mode (configurable via `holdEnterMs`) |
| Edit | Drag control | Ghost + drop indicator → release to place at grid position |
| Edit | Hold 2 s on control | Open context menu (configurable via `holdContextMs`) |
| Edit | Tap empty area | Exit edit mode |
| Edit | Press Escape | Exit edit mode (closes context menu first if open) |

**Visual cues in edit mode:**
- Controls show dashed outlines and get a `cursor: grab` style
- Subtle 45° diagonal stripe pattern on the grid background
- Locked controls: amber dashed outline + 6×6px amber pip at top-left corner
- Muted controls: 6×6px red pip at bottom-left corner (60% opacity)
- Floating banner at top: *"EDIT MODE — drag to reorder · hold for options · tap empty area or press ESC to exit"*
- The banner strobes briefly on entry (alternating accent-colored background)

**Note:** Both the lock indicator (amber pip) and mute indicator (red pip) are only visible in edit mode — they are hidden during normal usage.

**Interaction suppression:** In edit mode, native control actions (card toggle, slider drag, etc.) are suppressed via capturing-phase event interception. Only nInteractive's own drag/context-menu interactions fire.

### Context Menu

Available via 2 s hold in edit mode. Positioned at the pointer location, clamped to stay 8px from viewport edges.

| Option | Icon | Available On | Effect |
|---|---|---|---|
| LOCK POSITION | `ph-lock-key` | Unlocked controls | Pin the control to its current grid cell. It becomes an immovable obstacle during drag operations. Amber outline + pip indicator. |
| UNLOCK POSITION | `ph-lock-key-open` | Locked controls | Remove lock and pin — control reflows via CSS auto-placement. |
| MUTE SOUNDS | `ph-speaker-slash` | Unmuted controls | Suppress all `nbeep()` sounds for this control. Red pip indicator. Card toggles, slider drags, button clicks, etc. still work visually but produce no audio. |
| UNMUTE SOUNDS | `ph-speaker-high` | Muted controls | Restore sounds. |
| RESIZE (sub-menu) | `ph-resize` | All controls | Shows a grid of size buttons below the header. |
| CLOSE | `ph-x` | All | Dismiss the menu. |

**Resize sub-menu:** A 3-column grid of up to 9 size buttons appears:

| Button | Size | Label |
|---|---|---|
| `3×3` | 3 cols × 3 rows | XL |
| `3×2` | 3 cols × 2 rows | WIDE |
| `2×3` | 2 cols × 3 rows | TALL |
| `2×2` | 2 cols × 2 rows | FULL |
| `3×1` | 3 cols × 1 row | STRIP |
| `2×1` | 2 cols × 1 row | HALF-H |
| `1×3` | 1 col × 3 rows | PILLAR |
| `1×2` | 1 col × 2 rows | HALF-V |
| `1×1` | 1 col × 1 row | QUARTER |

- The current size is highlighted with an accent-colored active state.
- Sizes whose column span exceeds the current grid column count are hidden.
- Clicking a different size changes the control's `cols` and `rows` (and `data-size` for cards), then validates the existing pin position. If the new size doesn't fit at the old pin, the pin is removed and the control reflows.
- The grid rebuilds immediately after resize.

### Drag & Drop

Drag operations use a pin-based placement system with a collision bitmap:

1. **Start** (pointer down + 8px movement): A semi-transparent ghost clone follows the pointer. The original control dims. A drop indicator element appears. All non-locked pins are cleared. Grid geometry and a collision bitmap (of locked/pinned items) are cached for the duration of the drag.
2. **During drag:** The pointer position is mapped to a grid cell. `findPlacement()` runs against the cached bitmap to compute where the item will actually land (accounting for collision with locked items). The drop indicator shows:
   - **Green** (accent) dashed outline — valid placement position
   - **Amber** dashed outline — no valid position found (invalid drop)
3. **End** (pointer up): Locked items are re-pinned to their pre-drag snapshot positions. `resolvePin()` runs with a fresh bitmap to find the final placement. If valid, the item is pinned there. If no valid slot exists, a placement error animation fires (red flash + `nbeep('error')`) and the item returns to auto-flow.
4. **Cleanup:** Ghost and indicator are removed. Cached state is nulled.

**Key behavior:** Dragging never changes an item's size. The item lands at the same size it was before the drag. Only the resize sub-menu in the context menu can change card sizes.

### Lock System

Locked controls are immovable anchors in the grid:

- **When you lock a control:** Its current visual grid position is computed via centre-point cell mapping (`snapElToCell`). The position is collision-resolved and stored as a pin (`_config.pinned[idx]`). The control gets the `ni-locked` class.
- **During drag of other items:** Locked controls maintain their positions. Their pins survive the "clear stale pins" step at drag start. The drag bitmap includes them as obstacles. At drag end, locked positions are re-applied from the pre-drag snapshot.
- **In `sanitizePins()`:** Locked items are processed first (they win overlap ties). Non-locked items that collide with locked items are displaced or unpinned.
- **On unlock:** Both the lock and pin are removed — the control returns to auto-flow.
- **Visual:** Amber dashed outline + 6×6px amber pip (top-left), visible only in edit mode.

### Mute System

Muted controls still function normally (toggles toggle, sliders slide, buttons flash) but produce no audio:

- **How it works:** nDynamic's `wireInteractions()` includes an `isMuted(el)` helper that checks `el.closest('.ni-muted')`. Every `nbeep()` call for card toggles, button clicks, slider adjustments, toggle selections, and stepper increments is guarded by this check.
- **What is muted:** Card toggle sounds (`'on'`/`'off'`), button action sounds (`'action'`), slider adjustment sounds (`'adjust'`), toggle selection sounds (`'select'`), stepper sounds (`'increment'`/`'decrement'`).
- **What is NOT muted:** Dialog sounds (they come from a modal overlay, not the control element), edit-mode sounds (`'drag_start'`, `'drag_end'`, `'lock'`, `'resize'`, etc.).
- **Visual:** 6×6px red pip (bottom-left, 60% opacity), visible only in edit mode.

### Collision Detection

nInteractive uses a bitmap-based collision system to prevent overlapping pinned controls:

| Function | Purpose |
|---|---|
| `isBitmapFree(grid, col, row, rSpan, cSpan, cols)` | Checks if a rectangle of cells is entirely unoccupied |
| `occupyBitmap(grid, col, row, rSpan, cSpan)` | Marks a rectangle as occupied in the bitmap |
| `buildPinnedBitmap(excludeIdx)` | Builds an occupancy bitmap from all pinned items, optionally excluding one |
| `findNearestFreeSlot(grid, col, row, rSpan, cSpan, cols, maxSearch)` | Ring-expansion search (radius 1–30) for the nearest free rectangle, sorted by Euclidean distance within each ring |
| `findPlacement(grid, col, row, rSpan, cSpan, cols)` | Pure function: checks exact position, then falls back to nearest free slot. Used for drop indicator preview |
| `resolvePin(idx, col, row)` | Builds fresh bitmap excluding the target item, tries exact position, then nearest slot. Returns `{ col, row }` or `null`. Never auto-resizes |
| `sanitizePins()` | Full validation pass: processes locked items first, then non-locked. Displaces or drops pins that collide. Runs before every grid rebuild |

### CSS Classes Reference

Classes added/removed by nInteractive on DOM elements:

| Class | Applied To | Purpose |
|---|---|---|
| `ni-edit-mode` | Grid container | Active while in edit mode. Scopes all edit-mode visual styles |
| `ni-locked` | Control wrapper `[data-ni-idx]` | Marks a locked control |
| `ni-muted` | Control wrapper `[data-ni-idx]` | Marks a muted control. Also checked by nDynamic to suppress sounds |
| `ni-drag-ghost` | Cloned element (appended to body) | Semi-transparent drag ghost |
| `ni-drag-source` | Original control being dragged | Dimmed at 20% opacity during drag |
| `ni-drop-indicator` | Dynamic div (appended to body) | Shows where the dragged item will land |
| `ni-drop-invalid` | Drop indicator element | Added when no valid placement exists (amber variant) |
| `ni-placement-error` | Grid container | 600ms red flash animation when placement fails |
| `ni-edit-banner` | Banner div (appended to body) | Edit mode top banner |
| `ni-ctx-menu` | Context menu div | Context menu container |
| `ni-ctx-size-btn` | Size buttons in resize sub-menu | Individual size option. `.active` marks current size |

### Configuration (Interactive)

All nDynamic configuration options apply, plus:

| Key | Type | Default | Description |
|---|---|---|---|
| `holdEnterMs` | number | `5000` | Hold duration to enter edit mode from normal mode (ms) |
| `holdContextMs` | number | `2000` | Hold duration for context menu in edit mode (ms) |
| `onEditChange` | function | — | Callback `fn(isEditing)` fired on mode transitions |

Additional internal constants (not configurable):
- `HOLD_CANCEL_PX = 12` — pixel drift that cancels a normal-mode hold
- `DRAG_THRESHOLD = 8` — pixel movement that converts an edit-mode hold into a drag

### API (Interactive)

| Method | Description |
|---|---|
| `nInteractive.init(selector, controls, config?)` | Initialize the interactive layout. Calls `nDynamic.init()` internally |
| `nInteractive.destroy()` | Tear down all listeners, observers, and DOM. Calls `nDynamic.destroy()` |
| `nInteractive.isEditing()` | Returns `true` if currently in edit mode |
| `nInteractive.enterEdit()` | Programmatically enter edit mode |
| `nInteractive.exitEdit()` | Programmatically exit edit mode |
| `nInteractive.rebuild()` | Sanitize pins → update nDynamic → reapply control indices |
| `nInteractive.update(controls?, config?)` | Merge new controls/config and rebuild. Pass `null` to keep current |

---

## Procedural Audio (nbeep)

`nbeep.js` is a standalone procedural audio engine. Every sound is deterministically generated from a string seed — the same string always produces the same sound.

### Basic Usage

```js
// Play a single beep — deterministic from the string
nbeep('my_button_click');

// Play with pitch multiplier (used by sliders)
nbeep('adjust', false, 1.5);

// Play a looping tone (e.g., for alerts / dialogs)
nbeep('alarm_active', true);

// Stop all sound immediately
nDesignAudio.killActive();

// Any new nbeep() call auto-cancels the previous sound
nbeep('something_else');  // previous loop stops instantly
```

**AudioContext pre-warming:** Call `nDesignAudio.warmUp()` on a user gesture (`mousedown`/`touchstart`) to eliminate the 50–150ms latency from AudioContext resume on first sound. The interactive demo page does this automatically.

### Soundscapes

Four soundscapes are available, selected via `nDesignAudio.config.soundMode`:

| Mode | Key | Description |
|---|---|---|
| Retro | `'standard'` | Single-oscillator chirps. Scale-quantized sine/triangle tones with frequency slides. Classic sci-fi. |
| Harmonic | `'harmonic'` | Multi-voice chords. 3–5 layered oscillators with arpeggiation, vibrato, sub-octave ghosts, and shimmers. |
| nCARS | `'ncars'` | LCARS-style computer beeps. Pure sine sequences (1–4 notes) in the C5–A6 register. Clean, iconic. |
| nCARS 2 | `'ncars2'` | High-register LCARS variant (C6–C8). 20 patterns, dynamics profiles, micro-rests, dual-layer ghost notes. Brighter and more varied. **(Default)** |

Switch soundscape at runtime:

```js
nDesignAudio.config.soundMode = 'harmonic';
```

### Configuration (Audio)

All configuration is mutable at runtime via `nDesignAudio.config`:

| Property | Default | Description |
|---|---|---|
| `masterVolume` | `0.30` | Volume (0.0–1.0) |
| `maxDuration` | `0.10` | Duration scaling (0.0–1.0, scales all durations) |
| `soundMode` | `'ncars2'` | Soundscape: `'standard'`, `'harmonic'`, `'ncars'`, `'ncars2'` |
| `scale` | `'pentatonic'` | Musical scale (affects Retro and Harmonic modes). Options: `'pentatonic'`, `'minor_pentatonic'`, `'lydian'`, `'dorian'`, `'whole_tone'`, `'chromatic'` |
| `globalSeed` | `2026` | Changes entire soundscape deterministically |
| `useMusicalScale` | `true` | Use musical scale quantization |
| `durationMin` | `0.02` | Minimum duration in seconds (before `maxDuration` scaling) |
| `durationMax` | `0.30` | Maximum duration in seconds (before `maxDuration` scaling) |
| `allowedWaveforms` | `['sine', 'triangle']` | Array of allowed Web Audio waveforms |
| `fadeDuration` | `0.005` | Anti-pop envelope fade duration in seconds |

### Automatic Audio Integration

When both `nbeep.js` and `notum.js` are loaded, a delegated click listener on `document` automatically derives beep strings from element context:

| Element | Beep String |
|---|---|
| `[data-nbeep="X"]` | `X` (explicit) |
| `.tg-option` | `toggle_` + option value |
| `.stepper-btn` | `stepper_dec` or `stepper_inc` |
| `.demo-card` | `card_` + card name |
| `.dialog-btn` | `dialog_` + button value |
| `.submenu-item` | `menu_` + item name |
| `.seg-slider` | `slider_` + slider id |
| Any other `[data-flash]` | Trimmed text content (up to 64 chars) |

No per-component wiring needed. Every interactive element gets audio automatically.

**nDynamic integration:** When `wireInteractions()` runs (inside nDynamic), each control type gets individual `nbeep()` calls with specific sound keys:

| Control | Sound Key | When |
|---|---|---|
| Card | `'on'` / `'off'` | On toggle |
| Button | `'action'` | On click |
| Slider | `'adjust'` (pitch-scaled) | On value change during drag |
| Toggle | `'select'` | On option change |
| Stepper | `'increment'` / `'decrement'` | On +/− click |

All nDynamic sounds are suppressed if the control has the `ni-muted` CSS class (set by nInteractive's mute feature).

### Manual Audio Triggers

For custom elements not covered by the delegated listener:

```js
myButton.addEventListener('click', function() {
    nbeep('custom_action_name');
});

// Looping (e.g., while loading)
nbeep('loading_data', true);
// ... later ...
nDesignAudio.killActive();
```

---

## In-Grid Notifications (nNotify)

`nNotify.js` provides non-interactive, non-dialog notifications that display directly over the top row of any CSS Grid container. Unlike toasts, these notifications are rendered in-context — the top-row grid children are covered by a dithered grayout overlay (the same SVG checkerboard pattern used by dialogs), and a compact notification box is drawn centered inside the overlay. Notifications are rate-limited, queued, and auto-dismissed.

### Quick Start

```js
// Bind to a grid container
nNotify.init('#my-grid');

// Show a notification
nNotify.show({
  icon: 'ph-bell',
  title: 'SYSTEM READY',
  subtitle: 'All modules loaded',
  linger: 4000
});

// Programmatic dismiss
nNotify.dismiss();

// Tear down
nNotify.destroy();
```

### API Reference

| Method | Description |
|---|---|
| `nNotify.init(selectorOrElement)` | Bind to a grid container (works with nDynamic, nInteractive, or any CSS Grid) |
| `nNotify.show({ icon, title, subtitle, linger })` | Queue and display a notification. If rate-limited, queued and drained when cooldown expires |
| `nNotify.dismiss()` | Programmatically dismiss the current notification |
| `nNotify.destroy()` | Tear down overlay, clear queue, remove listeners |
| `nNotify.config({ rateLimit, linger, fade })` | Override default timing values |
| `nNotify.isActive()` | Returns `true` if a notification is currently displayed |
| `nNotify.queueLength()` | Number of queued notifications waiting to display |

### Configuration (Notifications)

| Option | Default | Description |
|---|---|---|
| `rateLimit` | `30000` | Minimum interval between notifications in milliseconds |
| `linger` | `4000` | Time in milliseconds before auto-dismiss |
| `fade` | `250` | Fade in/out transition duration in milliseconds |

### How It Works

1. **Top-row detection** — On `show()`, nNotify identifies all grid children whose top edge aligns with the container's first row and records their bounding rectangle.
2. **Overlay rendering** — An absolute-positioned `.nn-overlay` is placed over those top-row children. A `.nn-grayout` element covers the area with a dithered SVG checkerboard pattern (identical to the dialog grayout).
3. **Notification box** — A `.nn-box` is drawn centered within the overlay containing the icon, title, subtitle, and corner flasher anchors.
4. **Corner flashers** — All four corners (⌜ ⌝ ⌞ ⌟) strobe with 4 bursts of 50ms on/off, a 300ms pause, then repeat.
5. **Audio feedback** — If `nbeep` is available, a beep is triggered on show.
6. **Rate limiting** — Only one notification per `rateLimit` window. Calls during cooldown are pushed to a FIFO queue.
7. **Queue drain** — When cooldown expires, the next queued notification is shown automatically.
8. **Dismiss** — Tap/click anywhere on the overlay for instant dismiss, or wait for the `linger` timer. `nNotify.dismiss()` also works programmatically.

### CSS Classes (Notifications)

| Class | Element | Description |
|---|---|---|
| `.nn-overlay` | Container | Absolute-positioned overlay spanning the top-row grid children |
| `.nn-grayout` | Overlay child | Dithered SVG checkerboard mask over the top row |
| `.nn-box` | Notification | Centered box with `--surface` background, 1px border at 15% white, `--radius` border-radius |
| `.nn-icon` | Icon | Phosphor icon in `--accent` color, 1.25rem |
| `.nn-title` | Title | Rajdhani 700, 0.85rem, uppercase, 55% white opacity |
| `.nn-subtitle` | Subtitle | IBM Plex Mono, 0.7rem, `--text-dim` color |
| `.nn-content` | Wrapper | Flex container for title and subtitle |

### AHI Integration

When using the AHI protocol, call `notumAHI.notify()` to show an in-grid notification. If nNotify is not initialized, it falls back to a toast.

```js
notumAHI.notify({
  icon: 'ph-check-circle',
  title: 'COMPLETE',
  subtitle: 'Operation finished'
});
```

The JSON-RPC method `notify` accepts `{ icon, title, subtitle, linger }`. In flows, the `notify` step type displays a notification and resolves with `'notified'`.

---

## Interaction Feedback

All interactive elements should include `data-flash` to enable the unified press feedback.

### Flash Feedback (flashOutline)

**How it works:**
1. On click, the `.flash-outline` class rapidly toggles (10ms interval, 200ms total).
2. Re-clicks are blocked during the 200ms strobe (`WeakMap`-based lock).
3. Audio fires independently — the flash lock does not block `nbeep()`.

**Programmatic use:**

```js
// Flash any element — returns false if element is locked (mid-flash)
var accepted = flashOutline(myElement);

// With callback
flashOutline(myElement, function() {
    console.log('Flash done');
});
```

### Extended Lockout

Buttons can specify a longer lockout period using the `data-lockout` attribute. This extends the flash-lock duration beyond the default 200ms, preventing rapid re-clicks during longer operations.

```html
<!-- Lock button for 2 seconds after click -->
<div class="action-btn primary" data-flash data-lockout="2000">DEPLOY</div>

<!-- Lock for 5 seconds -->
<div class="action-btn danger" data-flash data-lockout="5000">EMERGENCY STOP</div>
```

**Behavior:**
- `data-lockout="<ms>"` — lockout duration in milliseconds
- When `lockout > 200ms`, the element gets the `.n-locked` CSS class during the lockout period
- When `lockout >= 400ms`, the strobe tick slows to 150ms (instead of 10ms) for a calmer visual rhythm
- The `flashOutline()` function returns `false` during lockout, blocking any re-invocation
- At the end of the lockout, `.n-locked` is removed and the element becomes clickable again

**Use cases:** Deploy buttons, destructive actions, operations that need time to complete, or anywhere you want to prevent accidental double-clicks.

---

## Icons

The framework uses [Phosphor Icons](https://phosphoricons.com/) (regular weight), self-hosted via `css/phosphor.css`.

```html
<i class="ph ph-lightbulb"></i>
<i class="ph ph-warning"></i>
<i class="ph ph-check-circle"></i>
<i class="ph ph-gear"></i>
```

Browse the full icon set at [phosphoricons.com](https://phosphoricons.com/).

---

## Color Reference

| Token | Hex | Usage |
|---|---|---|
| Background | `#0A0A0A` | App background |
| Surface | `#141414` | Cards, panels, dialogs |
| Text | `#F4F4F4` | Primary text (100% opacity) |
| Text Dim | `#F4F4F4` @ 40% | Secondary labels |
| Accent | `#00E5FF` | Active states, selections, data highlights |
| Amber | `#FFB300` | Warnings, locked indicators |
| Danger | `#FF3333` | Errors, off states, destructive actions, muted indicators |
| Border | `rgba(244,244,244, 0.10)` | Default borders |

CSS variables: `--bg`, `--surface`, `--accent`, `--danger`, `--amber`, `--radius` (2px).

---

## License

MIT License

Copyright (c) 2026 Notum Robotics

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
