/* ──────────────────────────────────────────────────────────────
   nDynamic — Intelligent Viewport-Filling Grid Layout Engine
   Copyright © 2026 Notum Robotics. Licensed under the MIT License.
   
   Automatically arranges Notum AHI controls into an
   optimal grid that fills the viewport without scrolling.
   
   USAGE:
     nDynamic.init('#my-container', controlArray, optionalConfig);
   
   CONTROL SCHEMA:
     Each control object in the array follows the same schema
     as GRID_CATALOG entries in notum.js:
       { type, cols, rows, size?, label?, icon?, state?, ... }
   
   CONFIG (all optional):
     {
       cols:      Number   — force column count (default: auto)
       rowHeight: Number   — base row height in px (default: auto-fit)
       gap:       Number   — grid gap in px (default: 6)
       padding:   Number   — container padding in px (default: 16)
       order:     [Number] — array of indices defining render order
       pinned:    { index: {col, row} } — pin specific items to positions
     }
   
   DEPENDENCY: nUtils.js must be loaded first.

   The engine:
   1. Measures viewport minus header/chrome
   2. Calculates optimal column count and row height
   3. Bin-packs controls using dense grid-auto-flow
   4. Resizes on viewport change via ResizeObserver
   5. Exposes nDynamic.rebuild() for manual refresh
   ────────────────────────────────────────────────────────────── */

var nDynamic = (function () {
    'use strict';

    /* ═══════════════════════════════════
       Internal State
       ═══════════════════════════════════ */

    var _container  = null;
    var _controls   = [];
    var _config     = {};
    var _observer   = null;
    var _debounceId = null;
    var _lastWidth  = 0;
    var _lastHeight = 0;
    var _resizeHandler = null;          // stored ref for cleanup
    var _docSliderListeners = null;     // single set of document-level slider listeners

    /* ═══════════════════════════════════
       Helpers (delegated to nUtils)
       ═══════════════════════════════════ */

    var escHtml     = nUtils.escHtml;
    var gridCorners = nUtils.gridCorners;

    /* ═══════════════════════════════════
       Segmented Bar Animations
       Canonical implementation — traveling dim square
       for partial fill, blink for full fill. Matches
       the main demo exactly.
       ═══════════════════════════════════ */

    function initSegBar(el) {
        // Clean up previous animation timers
        if (el._segTimer)   { clearInterval(el._segTimer);  el._segTimer = null; }
        if (el._segTimeout) { clearTimeout(el._segTimeout);  el._segTimeout = null; }

        var max   = parseInt(el.dataset.max) || 10;
        var value = parseInt(el.dataset.value) || 0;
        el.innerHTML = '';
        el._prevValue = value;  // Track for animated transitions
        el._segDir = 1;        // Animation direction: 1 = forward, -1 = reverse
        var isFull = (value >= max);
        if (isFull) el.classList.add('bar-full');
        else        el.classList.remove('bar-full');

        var filledSegs = [];
        for (var i = 0; i < max; i++) {
            var seg = document.createElement('div');
            seg.className = 'seg' + (i < value ? ' filled' : '');
            el.appendChild(seg);
            if (i < value) filledSegs.push(seg);
        }

        if (isFull && filledSegs.length > 1) {
            segBarBlink(el, filledSegs);
        } else if (filledSegs.length > 1) {
            segBarTravel(el, filledSegs);
        }
    }

    /**
     * updateSegBar — Animate a bar from its current value to a new value.
     * Decrease: fade-out the removed segments over 150ms then restart animation.
     * Increase: immediately show new segments with a brief bright flash.
     */
    function updateSegBar(el, newValue) {
        if (el._segTimer)    { clearInterval(el._segTimer);   el._segTimer = null; }
        if (el._segTimeout)  { clearTimeout(el._segTimeout);  el._segTimeout = null; }
        if (el._updateTimeout) { clearTimeout(el._updateTimeout); el._updateTimeout = null; }

        var max      = parseInt(el.dataset.max) || 10;
        var oldValue = el._prevValue !== undefined ? el._prevValue : (parseInt(el.dataset.value) || 0);
        newValue = Math.max(0, Math.min(max, newValue));
        el.dataset.value = newValue;
        el._prevValue = newValue;

        /* Determine animation direction from value delta */
        if (newValue !== oldValue) {
            el._segDir = newValue > oldValue ? 1 : -1;
        }

        var segs = el.querySelectorAll('.seg');
        if (!segs.length) { initSegBar(el); return; }

        var isFull = (newValue >= max);
        if (isFull) el.classList.add('bar-full');
        else        el.classList.remove('bar-full');

        if (newValue < oldValue) {
            /* ── Decrease: fade removed segments to the unfilled grey ── */
            for (var d = oldValue - 1; d >= newValue; d--) {
                if (segs[d]) {
                    segs[d].classList.remove('filled');
                }
            }
            /* After background transition completes, clean up and restart */
            el._updateTimeout = setTimeout(function () {
                el._updateTimeout = null;
                for (var i = 0; i < max; i++) {
                    segs[i].classList.remove('seg-dim', 'seg-bright', 'seg-flash');
                    if (i < newValue) segs[i].classList.add('filled');
                    else              segs[i].classList.remove('filled');
                }
                restartBarAnim(el, newValue, max);
            }, 350);

        } else if (newValue > oldValue) {
            /* ── Increase: show immediately, flash briefly ── */
            for (var j = 0; j < max; j++) {
                segs[j].classList.remove('seg-dim', 'seg-bright', 'seg-flash');
                if (j < newValue) segs[j].classList.add('filled');
                else              segs[j].classList.remove('filled');
            }
            /* Flash the newly added segments */
            for (var k = oldValue; k < newValue; k++) {
                if (segs[k]) segs[k].classList.add('seg-flash');
            }
            el._updateTimeout = setTimeout(function () {
                el._updateTimeout = null;
                for (var f = oldValue; f < newValue; f++) {
                    if (segs[f]) segs[f].classList.remove('seg-flash');
                }
                restartBarAnim(el, newValue, max);
            }, 350);
        }
        /* If value unchanged, do nothing */
    }

    function restartBarAnim(el, value, max) {
        if (el._segTimer)   { clearInterval(el._segTimer);  el._segTimer = null; }
        if (el._segTimeout) { clearTimeout(el._segTimeout);  el._segTimeout = null; }
        var segs = el.querySelectorAll('.seg');
        var filledSegs = [];
        for (var i = 0; i < value && i < segs.length; i++) {
            filledSegs.push(segs[i]);
        }
        if (value >= max && filledSegs.length > 1) {
            segBarBlink(el, filledSegs);
        } else if (filledSegs.length > 1) {
            segBarTravel(el, filledSegs);
        }
    }

    /* One darker square snaps across the filled segments, direction-aware.
       Direction (el._segDir): 1 = forward (left-to-right), -1 = reverse (right-to-left). */
    function segBarTravel(el, segs) {
        var dir = el._segDir || 1;
        var pos = dir === 1 ? 0 : segs.length - 1;
        segs[pos].classList.add('seg-dim');
        el._segTimer = setInterval(function () {
            var d = el._segDir || 1;
            segs[pos].classList.remove('seg-dim');
            pos = (pos + d + segs.length) % segs.length;
            segs[pos].classList.add('seg-dim');
        }, 120);
    }

    /* 2 flashes (50 ms on / 50 ms off) then 350 ms pause */
    function segBarBlink(el, segs) {
        var step = 0;
        function tick() {
            if (step < 4) {
                var on = (step % 2 === 0);
                for (var i = 0; i < segs.length; i++) {
                    if (on) segs[i].classList.add('seg-bright');
                    else    segs[i].classList.remove('seg-bright');
                }
                step++;
                el._segTimeout = setTimeout(tick, 50);
            } else {
                step = 0;
                el._segTimeout = setTimeout(tick, 350);
            }
        }
        tick();
    }

    /* ═══════════════════════════════════
       Dialog System (self-contained)
       Provides showDialog / closeDialog for any
       page that includes nDynamic + the dialog overlay HTML.
       ═══════════════════════════════════ */

    var _dialogResolve      = null;
    var _cornerStrobeTimer  = null;

    function showDialog(opts) {
        var $overlay   = document.getElementById('dialog-overlay');
        var $dialogBox = document.getElementById('dialog-box');
        if (!$overlay || !$dialogBox) {
            console.warn('[nDynamic] dialog-overlay / dialog-box not found in DOM');
            return Promise.resolve(null);
        }

        var flashOutline = window.flashOutline || function () { return true; };
        var FLASH_DURATION = 200;

        return new Promise(function (resolve) {
            _dialogResolve = resolve;

            var dialogBeepTitle = (opts.title || '').trim();
            var dialogBeepBody  = (opts.body  || '').trim();

            var html =
                '<span class="corner corner-tl">\u231C</span>' +
                '<span class="corner corner-tr">\u231D</span>' +
                '<span class="corner corner-bl">\u231E</span>' +
                '<span class="corner corner-br">\u231F</span>';
            if (opts.title) html += '<div class="dialog-title">' + escHtml(opts.title) + '</div>';
            if (opts.body)  html += '<div class="dialog-body">'  + escHtml(opts.body)  + '</div>';

            html += '<div class="dialog-actions">';
            (opts.buttons || []).forEach(function (btn) {
                var cls = 'dialog-btn';
                if (btn.style) cls += ' ' + btn.style;
                html += '<button class="' + cls + '" data-val="' + escHtml(btn.value) + '">' + escHtml(btn.label) + '</button>';
            });
            html += '</div>';

            $dialogBox.innerHTML = html;

            $dialogBox.querySelectorAll('.dialog-btn').forEach(function (el) {
                el.addEventListener('click', function () {
                    if (typeof flashOutline === 'function' && !flashOutline(el)) return;
                    if (typeof nbeep === 'function') {
                        nbeep('dialog_btn_' + (el.dataset.val || el.textContent));
                    }
                    setTimeout(function () { closeDialog(el.dataset.val); }, FLASH_DURATION);
                });
            });

            $overlay.classList.add('open');

            // Looping corner strobe: 5 flashes (50ms on/off), 200ms pause, repeat
            (function startCornerStrobe() {
                if (_cornerStrobeTimer) clearTimeout(_cornerStrobeTimer);
                var corners = $dialogBox.querySelectorAll('.corner');
                var flash = 0;
                var totalFlashes = 5;
                var on = false;

                function tick() {
                    if (!$overlay.classList.contains('open')) {
                        corners.forEach(function (c) { c.classList.remove('flash-outline'); });
                        _cornerStrobeTimer = null;
                        return;
                    }
                    if (flash < totalFlashes * 2) {
                        on = !on;
                        corners.forEach(function (c) {
                            if (on) c.classList.add('flash-outline');
                            else    c.classList.remove('flash-outline');
                        });
                        flash++;
                        _cornerStrobeTimer = setTimeout(tick, 50);
                    } else {
                        corners.forEach(function (c) { c.classList.remove('flash-outline'); });
                        _cornerStrobeTimer = setTimeout(function () {
                            flash = 0;
                            tick();
                        }, 200);
                    }
                }

                _cornerStrobeTimer = setTimeout(tick, 50);
            })();

            // Start looping beep (opts.alarm — default true)
            var playAlarm = opts.alarm !== undefined ? opts.alarm : true;
            if (playAlarm && typeof nbeep === 'function' && dialogBeepTitle) {
                nbeep(dialogBeepTitle, dialogBeepBody || true);
            }
        });
    }

    function closeDialog(value) {
        var $overlay   = document.getElementById('dialog-overlay');
        var $dialogBox = document.getElementById('dialog-box');

        if (_cornerStrobeTimer) {
            clearTimeout(_cornerStrobeTimer);
            _cornerStrobeTimer = null;
            if ($dialogBox) {
                $dialogBox.querySelectorAll('.corner').forEach(function (c) {
                    c.classList.remove('flash-outline');
                });
            }
        }
        if ($overlay) $overlay.classList.remove('open');
        if (typeof nDesignAudio === 'object' && nDesignAudio.killActive) {
            nDesignAudio.killActive();
        }
        if (_dialogResolve) {
            var fn = _dialogResolve;
            _dialogResolve = null;
            fn(value);
        }
    }

    // Allow overlay click to dismiss
    (function () {
        var $overlay = document.getElementById('dialog-overlay');
        if ($overlay) {
            $overlay.addEventListener('click', function (e) {
                if (e.target === $overlay) {
                    if (typeof nbeep === 'function') nbeep('dialog_dismiss');
                    closeDialog(null);
                }
            });
        }
    })();

    /* ═══════════════════════════════════
       Grid Dimension Calculator
       ═══════════════════════════════════ */

    /**
     * Compute optimal cols / rowHeight to fill the viewport
     * without scrolling, given the control catalog.
     */
    function computeGrid(containerW, availH, controls, cfg) {
        var gap     = cfg.gap || 6;
        var padding = cfg.padding || 16;

        // Effective area (subtract padding on both axes — box-sizing: border-box)
        var usableW = containerW - padding * 2;
        var usableH = availH - padding * 2;

        // Determine column count
        var cols;
        if (cfg.cols) {
            cols = cfg.cols;
        } else {
            // Auto: pick cols so a 1×1 cell is roughly square.
            // totalArea = sum of (item.cols × item.rows); with perfect packing
            // rows ≈ totalArea / cols.  Square cells ⟹ cols/rows ≈ W/H
            // ⟹ cols = √(totalArea × W/H).
            var totalArea = 0;
            for (var ci = 0; ci < controls.length; ci++) {
                totalArea += (controls[ci].cols || 1) * (controls[ci].rows || 1);
            }
            if (totalArea < 1) totalArea = 1;
            cols = Math.max(2, Math.round(Math.sqrt(totalArea * usableW / usableH)));
        }

        // Count total row-cells needed if laid out optimally
        // Simulate a simple greedy row-packing to estimate total rows
        var totalRows = estimateRows(controls, cols, cfg);

        // Calculate row height to fill available height
        var rowHeight;
        if (cfg.rowHeight) {
            rowHeight = cfg.rowHeight;
        } else {
            // rowHeight = (availH - (totalRows-1)*gap) / totalRows
            rowHeight = Math.max(40, Math.floor((usableH - (totalRows - 1) * gap) / totalRows));
            // Cap max row height for visual sanity
            rowHeight = Math.min(rowHeight, 120);
        }

        return { cols: cols, rowHeight: rowHeight, gap: gap, padding: padding, totalRows: totalRows };
    }

    /**
     * Estimate total grid rows consumed by greedy dense packing.
     * Uses a row-occupancy bitmap approach for accuracy.
     */
    function estimateRows(controls, cols, cfg) {
        var ordered = getOrderedControls(controls, cfg);
        // Bitmap: grid[row][col] = true if occupied
        var grid = [];
        var maxRow = 0;

        function isOpen(r, c, rowSpan, colSpan) {
            for (var dr = 0; dr < rowSpan; dr++) {
                for (var dc = 0; dc < colSpan; dc++) {
                    if (c + dc >= cols) return false;
                    if (grid[r + dr] && grid[r + dr][c + dc]) return false;
                }
            }
            return true;
        }

        function occupy(r, c, rowSpan, colSpan) {
            for (var dr = 0; dr < rowSpan; dr++) {
                if (!grid[r + dr]) grid[r + dr] = [];
                for (var dc = 0; dc < colSpan; dc++) {
                    grid[r + dr][c + dc] = true;
                }
            }
            var bottom = r + rowSpan;
            if (bottom > maxRow) maxRow = bottom;
        }

        for (var i = 0; i < ordered.length; i++) {
            var item = ordered[i];
            var cSpan = Math.min(item.cols || 1, cols);
            var rSpan = item.rows || 1;

            // Check pinned position
            if (cfg.pinned && cfg.pinned[item._origIdx] !== undefined) {
                var pin = cfg.pinned[item._origIdx];
                var pc = Math.max(0, Math.min(pin.col || 0, cols - cSpan));
                var pr = Math.max(0, pin.row || 0);
                /* Defensive: if pinned slot overlaps, fall through to auto-place */
                if (isOpen(pr, pc, rSpan, cSpan)) {
                    occupy(pr, pc, rSpan, cSpan);
                    continue;
                }
            }

            // Find first available slot (dense packing)
            var placed = false;
            for (var r = 0; !placed; r++) {
                for (var c = 0; c <= cols - cSpan; c++) {
                    if (isOpen(r, c, rSpan, cSpan)) {
                        occupy(r, c, rSpan, cSpan);
                        placed = true;
                        break;
                    }
                }
                if (r > 200) break; // Safety valve
            }
        }

        return maxRow;
    }

    /**
     * Return controls in render order, respecting cfg.order if set.
     */
    function getOrderedControls(controls, cfg) {
        var result = [];
        for (var i = 0; i < controls.length; i++) {
            var copy = {};
            for (var k in controls[i]) copy[k] = controls[i][k];
            copy._origIdx = i;
            result.push(copy);
        }
        if (cfg.order && Array.isArray(cfg.order)) {
            var ordered = [];
            for (var j = 0; j < cfg.order.length; j++) {
                var idx = cfg.order[j];
                if (idx >= 0 && idx < result.length) {
                    ordered.push(result[idx]);
                }
            }
            // Append any controls not in the order list
            var used = {};
            for (var m = 0; m < cfg.order.length; m++) used[cfg.order[m]] = true;
            for (var n = 0; n < result.length; n++) {
                if (!used[n]) ordered.push(result[n]);
            }
            return ordered;
        }
        return result;
    }

    /* ═══════════════════════════════════
       Render Individual Control
       ═══════════════════════════════════ */

    function renderControl(item, cols) {
        var el = document.createElement('div');
        var spanCol = Math.min(item.cols || 1, cols);
        var spanRow = item.rows || 1;

        switch (item.type) {
            case 'card': {
                var isOn = item.state === 'on';
                var cardSize = item.size || ((item.cols || 2) + 'x' + (item.rows || 2));
                el.className = 'card ' + (isOn ? 'is-on' : 'is-off') + ' nd-card';
                el.style.gridColumn = 'span ' + spanCol;
                el.style.gridRow = 'span ' + spanRow;
                el.dataset.on = item.on || 'ON';
                el.dataset.off = item.off || 'OFF';
                el.dataset.size = cardSize;
                el.innerHTML = gridCorners() +
                    '<div class="card-icon"><i class="ph ' + (item.icon || 'ph-circle') + '"></i></div>' +
                    '<div class="card-info">' +
                        '<div class="card-name">' + escHtml(item.name || '') + '</div>' +
                        '<div class="card-state">[ ' + escHtml(isOn ? (item.on || 'ON') : (item.off || 'OFF')) + ' ]</div>' +
                    '</div>';
                break;
            }
            case 'slider': {
                el.className = 'grid-cell gc-slider';
                el.style.gridColumn = 'span ' + spanCol;
                el.style.gridRow = 'span ' + spanRow;
                el.innerHTML = gridCorners() +
                    '<div class="slider-header">' +
                        '<span class="panel-label">' + escHtml(item.label || '') + '</span>' +
                        '<span class="slider-readout">[ ' + Math.round(((item.value || 0) / (item.max || 10)) * 100) + '% ]</span>' +
                    '</div>' +
                    '<div class="seg-slider" data-value="' + (item.value || 0) + '" data-max="' + (item.max || 10) + '" data-color="' + (item.color || 'accent') + '"></div>';
                break;
            }
            case 'toggle': {
                el.className = 'grid-cell gc-toggle';
                el.style.gridColumn = 'span ' + spanCol;
                el.style.gridRow = 'span ' + spanRow;
                var optHtml = '';
                (item.options || []).forEach(function (o, i) {
                    optHtml += '<div class="tg-option' + (i === item.active ? ' active' : '') + '" data-val="' + o + '">' + o + '</div>';
                });
                el.innerHTML = gridCorners() +
                    '<div class="panel-label">' + escHtml(item.label || '') + '</div>' +
                    '<div class="toggle-group">' + optHtml + '</div>';
                break;
            }
            case 'button': {
                el.className = 'action-btn grid-btn' + (item.style ? ' ' + item.style : '');
                el.style.gridColumn = 'span ' + spanCol;
                el.style.gridRow = 'span ' + spanRow;
                el.setAttribute('data-flash', '');
                if (item.lockout) {
                    el.dataset.lockout = item.lockout;
                }
                if (item.dialog) {
                    el.dataset.dialog = JSON.stringify(item.dialog);
                }
                el.innerHTML = (item.icon ? '<i class="ph ' + item.icon + '"></i>' : '') + '<span class="btn-label">' + escHtml(item.label || '') + '</span>';
                break;
            }
            case 'stepper': {
                el.className = 'grid-cell gc-stepper';
                el.style.gridColumn = 'span ' + spanCol;
                el.style.gridRow = 'span ' + spanRow;
                el.innerHTML = gridCorners() +
                    '<div class="panel-label">' + escHtml(item.label || '') + '</div>' +
                    '<div class="stepper">' +
                        '<div class="stepper-btn" data-dir="-1">\u2212</div>' +
                        '<div class="stepper-val">' + (item.value || 0) + '</div>' +
                        '<div class="stepper-btn" data-dir="+1">+</div>' +
                    '</div>';
                break;
            }
            case 'bar': {
                el.className = 'grid-cell gc-bar';
                el.style.gridColumn = 'span ' + spanCol;
                el.style.gridRow = 'span ' + spanRow;
                el.innerHTML =
                    '<div class="slider-header">' +
                        '<span class="panel-label">' + escHtml(item.label || '') + '</span>' +
                        '<span class="slider-readout">[ ' + Math.round(((item.value || 0) / (item.max || 10)) * 100) + '% ]</span>' +
                    '</div>' +
                    '<div class="seg-bar" data-value="' + (item.value || 0) + '" data-max="' + (item.max || 10) + '" data-color="' + (item.color || 'accent') + '"></div>';
                break;
            }
            case 'status': {
                el.className = 'grid-cell gc-status';
                el.style.gridColumn = 'span ' + spanCol;
                el.style.gridRow = 'span ' + spanRow;
                var rows = '';
                (item.items || []).forEach(function (r) {
                    rows += '<div class="status-row"><span class="status-label">' + escHtml(r.k || '') +
                            '</span><span class="status-val' + (r.c ? ' ' + r.c : '') + '">' +
                            escHtml(r.v || '') + '</span></div>';
                });
                el.innerHTML = gridCorners() +
                    '<div class="panel-label">' + escHtml(item.label || '') + '</div>' +
                    '<div class="status-grid">' + rows + '</div>';
                break;
            }
        }
        return el;
    }

    /* ═══════════════════════════════════
       Interactive Behavior Wiring
       ═══════════════════════════════════ */

    function wireInteractions(container) {
        var flashOutline = window.flashOutline || function () { return true; };

        /* Check if an element (or its grid-item ancestor) is muted by nInteractive */
        function isMuted(el) {
            return !!(el && el.closest && el.closest('.ni-muted'));
        }

        /**
         * Play a beep respecting per-control hash suffix and soundscape theme.
         * Reads data-ni-beep (hash suffix 001-999) and data-ni-theme (soundMode)
         * from the nearest grid-cell ancestor.
         */
        function beep(el, text, loop, pitch) {
            if (typeof nbeep !== 'function' || isMuted(el)) return;
            var cell = el.closest && el.closest('[data-ni-beep],[data-ni-theme]');
            var suffix = cell && cell.dataset.niBeep ? cell.dataset.niBeep : '';
            var theme  = cell && cell.dataset.niTheme ? cell.dataset.niTheme : '';
            var oldMode;
            if (theme && typeof nDesignAudio !== 'undefined') {
                oldMode = nDesignAudio.config.soundMode;
                nDesignAudio.config.soundMode = theme;
            }
            nbeep(suffix ? text + suffix : text, loop, pitch);
            if (oldMode !== undefined) nDesignAudio.config.soundMode = oldMode;
        }

        // Card toggles
        container.querySelectorAll('.nd-card').forEach(function (card) {
            card.addEventListener('click', function () {
                if (typeof flashOutline === 'function' && !flashOutline(card)) return;
                var isOn = card.classList.contains('is-on');
                var onLabel  = card.dataset.on || 'ON';
                var offLabel = card.dataset.off || 'OFF';
                card.classList.remove('is-on', 'is-off');
                card.classList.add(isOn ? 'is-off' : 'is-on');
                var stateEl  = card.querySelector('.card-state');
                if (stateEl) stateEl.textContent = isOn ? '[ ' + offLabel + ' ]' : '[ ' + onLabel + ' ]';
                beep(card, isOn ? 'off' : 'on');
            });
        });

        // Button flash + optional dialog
        container.querySelectorAll('[data-flash]').forEach(function (el) {
            el.addEventListener('click', function () {
                if (typeof flashOutline === 'function') flashOutline(el);
                beep(el, 'action');
                // If button carries a dialog definition, open it
                if (el.dataset.dialog) {
                    try {
                        var opts = JSON.parse(el.dataset.dialog);
                        showDialog(opts);
                    } catch (_) {}
                }
            });
        });

        // Slider init
        // Clean up any prior document-level slider listeners to avoid accumulation
        if (_docSliderListeners) {
            document.removeEventListener('mousemove', _docSliderListeners.onMove);
            document.removeEventListener('touchmove', _docSliderListeners.onTouchMove);
            document.removeEventListener('mouseup',   _docSliderListeners.onUp);
            document.removeEventListener('touchend',  _docSliderListeners.onUp);
            _docSliderListeners = null;
        }

        var _activeSliderDrag = null; // { el, max, readout, lastSoundedValue }

        function sliderInteraction(e) {
            if (!_activeSliderDrag) return;
            var d   = _activeSliderDrag;
            var rect = d.el.getBoundingClientRect();
            var x = (e.touches ? e.touches[0].clientX : e.clientX) - rect.left;
            var segW = rect.width / d.max;
            var idx = Math.max(0, Math.min(d.max - 1, Math.floor(x / segW)));
            var newVal = idx + 1;
            d.el.dataset.value = newVal;
            d.el.querySelectorAll('.seg').forEach(function (s, si) {
                s.classList.toggle('filled', si < newVal);
            });
            if (d.readout) d.readout.textContent = '[ ' + Math.round((newVal / d.max) * 100) + '% ]';
            if (newVal !== d.lastSoundedValue) {
                d.lastSoundedValue = newVal;
                var pct = newVal / d.max;
                var pitchMultiplier = Math.pow(2, (pct - 0.5) * 2);
                beep(d.el, 'adjust', false, pitchMultiplier);
            }
        }

        _docSliderListeners = {
            onMove:      function (e) { if (_activeSliderDrag) sliderInteraction(e); },
            onTouchMove: function (e) { if (_activeSliderDrag) sliderInteraction(e); },
            onUp:        function ()  { _activeSliderDrag = null; }
        };
        document.addEventListener('mousemove', _docSliderListeners.onMove);
        document.addEventListener('touchmove', _docSliderListeners.onTouchMove, { passive: true });
        document.addEventListener('mouseup',   _docSliderListeners.onUp);
        document.addEventListener('touchend',  _docSliderListeners.onUp);

        container.querySelectorAll('.seg-slider').forEach(function (el) {
            var max   = parseInt(el.dataset.max) || 10;
            var value = parseInt(el.dataset.value) || 0;
            var readout = el.parentNode ? el.parentNode.querySelector('.slider-readout') : null;
            el.innerHTML = '';
            for (var i = 0; i < max; i++) {
                var seg = document.createElement('div');
                seg.className = 'seg' + (i < value ? ' filled' : '');
                seg.dataset.idx = i;
                el.appendChild(seg);
            }
            el.addEventListener('mousedown', function (e) {
                _activeSliderDrag = { el: el, max: max, readout: readout, lastSoundedValue: -1 };
                sliderInteraction(e);
            });
            el.addEventListener('touchstart', function (e) {
                _activeSliderDrag = { el: el, max: max, readout: readout, lastSoundedValue: -1 };
                sliderInteraction(e);
            }, { passive: true });
        });

        // Bar init (read-only, with animations matching main demo)
        container.querySelectorAll('.seg-bar').forEach(function (el) {
            initSegBar(el);
        });

        // Toggle groups
        container.querySelectorAll('.toggle-group').forEach(function (group) {
            group.querySelectorAll('.tg-option').forEach(function (opt) {
                opt.addEventListener('click', function () {
                    if (opt.classList.contains('active')) return;
                    if (typeof flashOutline === 'function' && !flashOutline(opt)) return;
                    var cur = group.querySelector('.tg-option.active');
                    if (cur) cur.classList.remove('active');
                    opt.classList.add('active');
                    beep(opt, 'select');
                });
            });
        });

        // Steppers
        container.querySelectorAll('.stepper').forEach(function (el) {
            var valEl = el.querySelector('.stepper-val');
            var val = parseInt(valEl.textContent) || 0;
            el.querySelectorAll('.stepper-btn').forEach(function (btn) {
                btn.addEventListener('click', function () {
                    if (typeof flashOutline === 'function' && !flashOutline(btn)) return;
                    var dir = parseInt(btn.dataset.dir);
                    val = Math.max(0, val + dir);
                    valEl.textContent = val;
                    beep(btn, dir > 0 ? 'increment' : 'decrement');
                });
            });
        });
    }

    /* ═══════════════════════════════════
       Build / Rebuild
       ═══════════════════════════════════ */

    function build() {
        if (!_container || !_controls.length) return;

        var containerRect = _container.getBoundingClientRect();
        var containerW = containerRect.width;

        // Available height: prefer the container's own height (set by flex layout),
        // fall back to viewport-based calculation if container has no definite height yet.
        var viewH  = window.innerHeight;
        var availH = containerRect.height > 50
            ? containerRect.height
            : viewH - containerRect.top - (_config.padding || 16);
        if (availH < 200) availH = 200;

        var dims = computeGrid(containerW, availH, _controls, _config);

        // Apply grid styles
        _container.style.display = 'grid';
        _container.style.gridTemplateColumns = 'repeat(' + dims.cols + ', 1fr)';
        _container.style.gridAutoRows = dims.rowHeight + 'px';
        _container.style.gridAutoFlow = 'dense';
        _container.style.gap = dims.gap + 'px';
        _container.style.padding = dims.padding + 'px';
        _container.style.boxSizing = 'border-box';
        _container.style.width = '100%';
        _container.style.maxHeight = availH + 'px';
        _container.style.overflow = 'hidden';

        // Clear and re-render
        _container.innerHTML = '';

        var ordered = getOrderedControls(_controls, _config);
        var frag = document.createDocumentFragment();

        for (var i = 0; i < ordered.length; i++) {
            var item = ordered[i];
            var el = renderControl(item, dims.cols);
            if (el) {
                /* Apply explicit grid position for pinned items */
                if (_config.pinned && _config.pinned[item._origIdx] !== undefined) {
                    var pin = _config.pinned[item._origIdx];
                    var pCol = Math.max(0, pin.col || 0);
                    var pRow = Math.max(0, pin.row || 0);
                    /* Clamp to grid bounds */
                    var itemCols = Math.min(item.cols || 1, dims.cols);
                    var itemRows = item.rows || 1;
                    pCol = Math.min(pCol, dims.cols - itemCols);
                    /* Use full shorthand so span is preserved alongside start position */
                    el.style.gridColumn = (pCol + 1) + ' / span ' + itemCols;
                    el.style.gridRow    = (pRow + 1) + ' / span ' + itemRows;
                }
                frag.appendChild(el);
            }
        }

        _container.appendChild(frag);

        // Wire all interactions
        wireInteractions(_container);

        _lastWidth  = containerW;
        _lastHeight = viewH;
    }

    function rebuild() {
        // Debounced rebuild (16ms = one frame)
        if (_debounceId) cancelAnimationFrame(_debounceId);
        _debounceId = requestAnimationFrame(function () {
            _debounceId = null;
            build();
        });
    }

    /* ═══════════════════════════════════
       Public API
       ═══════════════════════════════════ */

    /**
     * nDynamic.init(selector, controls, config?)
     *
     * @param {string|Element} selector  CSS selector or DOM element
     * @param {Array}          controls  Array of control definition objects
     * @param {Object}         config    Optional configuration overrides
     */
    function init(selector, controls, config) {
        // Resolve container
        if (typeof selector === 'string') {
            _container = document.querySelector(selector);
        } else {
            _container = selector;
        }

        if (!_container) {
            console.error('[nDynamic] Container not found:', selector);
            return;
        }

        _controls = controls || [];
        _config   = config || {};

        // Initial build
        build();

        // Watch for resize
        if (_observer) _observer.disconnect();

        if (typeof ResizeObserver !== 'undefined') {
            _observer = new ResizeObserver(function (entries) {
                // Only rebuild if dimensions meaningfully changed (>10px)
                var entry = entries[0];
                if (!entry) return;
                var w = entry.contentRect.width;
                var h = window.innerHeight;
                if (Math.abs(w - _lastWidth) > 10 || Math.abs(h - _lastHeight) > 10) {
                    rebuild();
                }
            });
            _observer.observe(_container);
        }

        // Also watch viewport height changes
        if (_resizeHandler) window.removeEventListener('resize', _resizeHandler);
        _resizeHandler = function () {
            var h = window.innerHeight;
            if (Math.abs(h - _lastHeight) > 10) {
                rebuild();
            }
        };
        window.addEventListener('resize', _resizeHandler);
    }

    /**
     * nDynamic.destroy()
     * Tear down the dynamic grid and observers.
     */
    function destroy() {
        if (_observer) {
            _observer.disconnect();
            _observer = null;
        }
        if (_resizeHandler) {
            window.removeEventListener('resize', _resizeHandler);
            _resizeHandler = null;
        }
        if (_docSliderListeners) {
            document.removeEventListener('mousemove', _docSliderListeners.onMove);
            document.removeEventListener('touchmove', _docSliderListeners.onTouchMove);
            document.removeEventListener('mouseup',   _docSliderListeners.onUp);
            document.removeEventListener('touchend',  _docSliderListeners.onUp);
            _docSliderListeners = null;
        }
        if (_container) {
            _container.innerHTML = '';
            _container.removeAttribute('style');
        }
        _controls = [];
        _config   = {};
    }

    /**
     * nDynamic.update(controls?, config?)
     * Update controls and/or config and rebuild.
     */
    function update(controls, config) {
        if (controls) _controls = controls;
        if (config) {
            for (var k in config) _config[k] = config[k];
        }
        rebuild();
    }

    // Expose public API
    return {
        init:         init,
        rebuild:      rebuild,
        destroy:      destroy,
        update:       update,
        showDialog:   showDialog,
        closeDialog:  closeDialog,
        updateSegBar: updateSegBar
    };

})();
